import pathlib
import syside

EXAMPLE_DIR = pathlib.Path(__file__).parent
MODEL_FILE_PATH = EXAMPLE_DIR / "example_model.sysml"


def build_explicit_subset_list(
    element: syside.PartUsage,
    part_def: syside.PartDefinition,
    subsetting_collection: syside.PartUsage,
) -> None:
    """
    Recursively builds explicit subset lists for parts using the subsetting pattern.

    Creates an in-memory explicit list from subsetting relationships:
        part <element> subsets <subsetting_collection> { ... }
    becomes:
        part redefines <subsetting_collection> = (<element>, ...)

    Args:
        element: SysML element to process
        part_def: Part definition type to match when filtering
        subsetting_collection: The feature being subsetted
    """

    if (
        part_def not in element.part_definitions.collect()
        or element.name == subsetting_collection.name
    ):
        return

    build_list = True

    # Check if subsetting part has been explicitly redefined with a value
    collection_name = subsetting_collection.name
    if collection_name is not None:
        subpart_usage = element.get_member(collection_name)
        if subpart_usage is not None and hasattr(
            subpart_usage, "feature_value_expression"
        ):
            compiler = syside.Compiler()
            subpart_value, _ = compiler.evaluate(
                subpart_usage.feature_value_expression
            )
            if subpart_value is not None:
                build_list = False

    # Build list if subpart value is not defined
    if build_list:
        list_items = [
            x
            for x in element.members.collect()
            if isinstance(x, syside.PartUsage)
            and subsetting_collection
            in [
                y.subsetted_feature
                for y in x.heritage.relationships
                if type(y) is syside.Subsetting
            ]
        ]

        # Only create the explicit list if we found subsetting parts
        if len(list_items) > 0:
            # Create a new redefinition of subsetting_collection
            _, subpart = element.children.append(
                syside.FeatureMembership, syside.PartUsage
            )
            subpart.heritage.append(syside.Redefinition, subsetting_collection)

            # Create a comma-separated list expression
            _, operator_exp = subpart.feature_value_member.set_member_element(
                syside.OperatorExpression
            )

            operator_exp.try_set_operator(syside.ExplicitOperator.Comma)

            # Add each part that subsets the collection to the list
            for item in list_items:
                _, expr = operator_exp.arguments.append(
                    syside.FeatureReferenceExpression
                )
                expr.referent_member.set_member_element(item)

    # Recursively process all child parts
    for child in element.members.collect():
        if type(child) is syside.PartUsage:
            build_explicit_subset_list(child, part_def, subsetting_collection)


def rollup_values(
    document: syside.Document,
    stdlib: syside.Stdlib,
    package_name: str,
    rollup_part_def: str,
    subparts_collection_name: str,
    rollup_attribute_name: str,
    root_element_name: str,
) -> syside.Value | None:
    try:
        package = document.root_node[package_name].cast(syside.Package)
        rollup_type = package[rollup_part_def].cast(syside.PartDefinition)
        subparts_collection = rollup_type[subparts_collection_name].cast(
            syside.PartUsage
        )
        rollup_attribute = rollup_type[rollup_attribute_name].cast(
            syside.Feature
        )
        root_element = package[root_element_name].cast(syside.PartUsage)
    except KeyError as err:
        print(
            f"At least one of the required elements do not exist in the {package_name} definitions: {err}"
        )
        exit(1)

    # Build explicit subset lists for all parts that use the subsetting pattern
    build_explicit_subset_list(
        element=root_element,
        part_def=rollup_type,
        subsetting_collection=subparts_collection,
    )

    # Evaluate the expression with automatic unit conversion
    compiler = syside.Compiler()
    value, report = compiler.evaluate_feature(
        feature=rollup_attribute,
        scope=root_element,
        stdlib=stdlib,
        experimental_quantities=True,  # Enables automatic unit conversion
    )

    if report.fatal:
        print(report.diagnostics)
        exit(1)

    return value


def main() -> None:
    model, _ = syside.load_model(paths=[MODEL_FILE_PATH])

    with model.user_docs[0].lock() as locked:
        mass_rollup_result = rollup_values(
            document=locked,
            stdlib=model.environment.lib,
            package_name="VehicleMassRollup",
            rollup_part_def="MassedThing",
            subparts_collection_name="subcomponents",
            rollup_attribute_name="totalMass",
            root_element_name="vehicle",
        )
        # Evaluates to 1,800.00 kg
        print(f"Calculated total mass: {mass_rollup_result:,.2f} kg")

        power_rollup_result = rollup_values(
            document=locked,
            stdlib=model.environment.lib,
            package_name="AircraftPowerRollup",
            rollup_part_def="PoweredComponent",
            subparts_collection_name="poweredSubsystems",
            rollup_attribute_name="totalPower",
            root_element_name="aircraft",
        )
        # Evaluates to 1,405.48 W
        print(f"Calculated total power: {power_rollup_result:,.2f} W")


if __name__ == "__main__":
    main()
