import pathlib
from dataclasses import dataclass

import syside

EXAMPLE_DIR = pathlib.Path(__file__).parent
MODEL_FILE_PATH = EXAMPLE_DIR / "example_model.sysml"
STANDARD_LIBRARY = syside.Environment.get_default().lib

# Filters in these packages are displayed as trees, the rest as flat lists
TREE_VIEWS = {"ElectricalComponents", "MechanicalComponents"}


def find_element_by_name(
    model: syside.Model, name: str
) -> syside.Element | None:
    """Search the model for a specific element by name."""

    for element in model.elements(syside.Element, include_subtypes=True):
        if element.name == name:
            return element
    return None


def get_filter_expressions(
    model: syside.Model,
) -> dict[syside.Package, list[syside.Expression]]:
    """Extract filter expressions grouped by their owning package."""
    filter_groups: dict[syside.Package, list[syside.Expression]] = {}

    for efm in model.elements(syside.ElementFilterMembership):
        package = efm.parent
        if not isinstance(package, syside.Package):
            continue

        assert efm.condition is not None

        if package not in filter_groups:
            filter_groups[package] = []

        filter_groups[package].append(efm.condition)

    return filter_groups


def matches_filter(
    compiler: syside.Compiler,
    element: syside.Element,
    filter_expr: syside.Expression,
) -> bool:
    """Check if an element matches the filter."""

    result, compilation_report = compiler.evaluate_filter(
        target=element,
        filter=filter_expr,
        stdlib=STANDARD_LIBRARY,
    )
    if compilation_report.fatal:
        print(compilation_report.diagnostics)
        exit(1)

    return bool(result)


def collect_matching(
    compiler: syside.Compiler,
    element: syside.Element,
    filter_exprs: list[syside.Expression],
) -> list[syside.Element]:
    """Collect all PartUsage descendants that match all filter expressions."""
    matched = []

    if all(matches_filter(compiler, element, expr) for expr in filter_exprs):
        matched.append(element)

    for child in element.owned_elements.collect():
        if isinstance(child, syside.PartUsage):
            matched.extend(collect_matching(compiler, child, filter_exprs))

    return matched


@dataclass
class FilteredNode:
    name: str
    children: list["FilteredNode"]


def build_filtered_tree(
    element: syside.Namespace,
    matched_elements: list[syside.Element],
) -> FilteredNode | None:
    """Build a tree containing only matching elements, preserving hierarchy among them. Non-matching ancestors are included for context."""
    self_matches = element in matched_elements

    children = []
    for child in element.owned_members.collect():
        if isinstance(child, syside.PartUsage):
            subtree = build_filtered_tree(child, matched_elements)
            if subtree is not None:
                children.append(subtree)

    if self_matches or children:
        name = element.name or "<anonymous>"
        return FilteredNode(
            name=name if self_matches else "(" + name + ")",
            children=children,
        )
    return None


def print_tree(
    node: FilteredNode, prefix: str = "", is_last: bool = True
) -> None:
    """Print a tree using box-drawing characters."""
    connector = " └── " if is_last else " ├── "
    print(f"{prefix}{connector}{node.name}")

    child_prefix = prefix + ("     " if is_last else " │   ")
    for i, child in enumerate(node.children):
        print_tree(child, child_prefix, i == len(node.children) - 1)


def main() -> None:
    # Load SysML model and get diagnostics (errors/warnings)
    (model, _) = syside.load_model([MODEL_FILE_PATH], warnings_as_errors=True)

    filters = get_filter_expressions(model)

    vehicle = find_element_by_name(model, "vehicle")
    assert vehicle is not None and isinstance(vehicle, syside.PartDefinition)

    compiler = syside.Compiler()

    for package, filter_exprs in filters.items():
        view_name = "<anonymous>"

        if package.name:
            view_name = str(package.name)
        elif isinstance(package.parent, syside.Import):
            view_name = "<filtered import>"

        matched = collect_matching(compiler, vehicle, filter_exprs)

        if view_name in TREE_VIEWS:
            # Tree view for presence-based filters
            tree = build_filtered_tree(vehicle, matched)
            print(f"\n{view_name}")
            if tree is not None:
                for i, child in enumerate(tree.children):
                    print_tree(child, "", i == len(tree.children) - 1)
        else:
            # Flat list for value-based filters
            names = [p.name for p in matched if p.name]
            print(f"{view_name}: {', '.join(names) if names else '(none)'}")


if __name__ == "__main__":
    main()
