import pathlib
import syside

EXAMPLE_DIR = pathlib.Path(__file__).parent
MODEL_FILE_PATH = EXAMPLE_DIR / "example_model.sysml"
STANDARD_LIBRARY = syside.Environment.get_default().lib


def find_element_by_name(
    model: syside.Model, name: str
) -> syside.Element | None:
    """Search the model for a specific element by name."""

    # Iterates through all model elements that subset Element type
    # e.g. PartUsage, ItemUsage, OccurrenceUsage, etc.
    for element in model.elements(syside.Element, include_subtypes=True):
        if element.name == name:
            return element
    return None


def deduplicate_attributes(
    attributes: list[syside.AttributeUsage],
) -> list[syside.Feature]:
    """Removes attributes that have been redefined, keeping only the active versions.

    Args:
        attributes: List of attributes to deduplicate

    """
    redefined = set()

    for attribute in attributes:
        for inherited_relationship in attribute.heritage.relationships:
            if isinstance(inherited_relationship, syside.Redefinition):
                redefined.add(inherited_relationship.first_target)

    # Only keep attributes that are not redefined by anything
    return [attr for attr in attributes if attr not in redefined]


def show_part_attributes(part: syside.PartUsage, level: int = 0) -> None:
    """Prints attributes and their evaluated values for a part.

    Args:
        part: The part whose attributes to display

        level: Indentation level for hierarchical display

    """
    # Filter to get only user-defined attributes (exclude standard library)
    filtered_attributes = deduplicate_attributes(
        [
            x
            for x in part.usages.collect()
            if type(x) is syside.AttributeUsage
            and x.document.document_tier is syside.DocumentTier.Project
        ]
    )

    for attribute in filtered_attributes:
        # Evaluate the attribute value in the context of the part
        value = evaluate_feature(attribute, part)
        # For enum values, display just the name
        if type(value) is syside.EnumerationUsage:
            value = value.name
        print("  " * level, f" └ Attribute: {attribute.name} = {value}")


def evaluate_feature(
    feature: syside.Feature, scope: syside.Type
) -> syside.Value | None:
    """Evaluates a feature within a given scope.

    Args:
        feature: The feature to evaluate (attribute, part, etc.)

        scope: The context in which to evaluate the feature

    """
    compiler = syside.Compiler()
    value, compilation_report = compiler.evaluate_feature(
        feature=feature,
        scope=scope,
        stdlib=STANDARD_LIBRARY,
        experimental_quantities=True,
    )
    if compilation_report.fatal:
        print(compilation_report.diagnostics)
        exit(1)
    return value


def walk_ownership_tree(element: syside.PartUsage, level: int = 0) -> None:
    """Recursively prints the part hierarchy with attributes in a tree format.

    Args:
        element: The part to display and traverse

        level: Indentation level for hierarchical display

    """

    # Skip printing root node name
    if level > 0:
        if element.name is not None:
            print("  " * level, f"Part: {element.name}")
        else:
            print("  " * level, "Part: <anonymous>")

    # Get attributes and their values
    show_part_attributes(element, level)

    # Filter child parts: exclude library elements
    filtered_children = [
        x
        for x in element.usages.collect()
        if type(x) is syside.PartUsage
        and x.document.document_tier is syside.DocumentTier.Project
    ]

    # Recursively process each child part
    for child in filtered_children:
        walk_ownership_tree(child, level + 1)


def main() -> None:
    # Load SysML model and get diagnostics (errors/warnings)
    (model, _) = syside.load_model([MODEL_FILE_PATH], warnings_as_errors=True)

    # Find the variation definition containing all configurations
    available_configurations = find_element_by_name(
        model, "AvailableConfigurations"
    )
    assert available_configurations is not None and isinstance(
        available_configurations, syside.PartDefinition
    )

    # For each variant configuration, print its part hierarchy and attributes
    for index, config in enumerate(available_configurations.variants.collect()):
        if type(config) is syside.PartUsage:
            print(
                "\n",
                "=" * 40,
                f"\n CONFIGURATION #{index + 1}: {config.name}\n",
                "=" * 40,
            )
            walk_ownership_tree(config)
    print()


if __name__ == "__main__":
    main()
