import pathlib
import syside

EXAMPLE_DIR = pathlib.Path(__file__).parent
MODEL_FILE_PATH = EXAMPLE_DIR / "example_model.sysml"
STANDARD_LIBRARY = syside.Environment.get_default().lib


def find_element_by_name(
    model: syside.Model, name: str
) -> syside.Element | None:
    """Search the model for a specific element by name."""
    for element in model.elements(syside.Element, include_subtypes=True):
        if element.name == name:
            return element
    return None


def evaluate_feature(
    feature: syside.Feature, scope: syside.Type
) -> syside.Value | None:
    """Evaluate a feature within a given scope.

    Args:
        feature: The feature to evaluate (attribute, calc usage, etc.)

        scope: The context in which to evaluate the feature

    """
    compiler = syside.Compiler()
    value, compilation_report = compiler.evaluate_feature(
        feature=feature,
        scope=scope,
        stdlib=STANDARD_LIBRARY,
        experimental_quantities=True,
    )
    if compilation_report.fatal:
        print(compilation_report.diagnostics)
        exit(1)
    return value


def main() -> None:
    # Load SysML model and get diagnostics (errors/warnings)
    (model, _) = syside.load_model([MODEL_FILE_PATH], warnings_as_errors=True)

    steel_beam = find_element_by_name(model, "SteelBeam")
    assert steel_beam is not None and isinstance(
        steel_beam, syside.PartDefinition
    )

    attributes = [
        x
        for x in steel_beam.usages.collect()
        if type(x) is syside.AttributeUsage
        and x.document.document_tier is syside.DocumentTier.Project
    ]

    for attr in attributes:
        value = evaluate_feature(attr, steel_beam)
        # Only print scalar values, skip complex objects like material references
        if isinstance(value, (int, float)):
            print(f"  {attr.name} = {value:.4f}")


if __name__ == "__main__":
    main()
